/*:
 * @target MZ
 * @plugindesc (Compat) Try to force VisuStella Quest Journal to open on the "Active" tab.
 * @help
 * Places a small compatibility shim that, when Scene_Quest is pushed,
 * attempts several safe methods to switch the journal to the Active tab.
 */

(() => {
  const _SceneManager_push = SceneManager.push;
  SceneManager.push = function(sceneClass) {
    _SceneManager_push.call(this, sceneClass);

    try {
      // If the quest scene is being opened, attempt to set the default/category
      if (typeof Scene_Quest !== 'undefined' && sceneClass === Scene_Quest) {
        // let the scene initialize, then try to switch the tab
        setTimeout(() => {
          const scn = SceneManager._scene;
          if (!scn) return;

          // Method A: legacy-style - set a gameSystem flag (harmless if unused)
          try { $gameSystem._questJournalCategory = 'active'; } catch (e) {}

          // Method B: try calling likely methods on the scene's windows/objects
          const tryNames = ['setCategory','changeCategory','selectCategory','openCategory','selectTab','setTab','changeTab'];
          let switched = false;

          for (const prop in scn) {
            if (!Object.prototype.hasOwnProperty.call(scn, prop)) continue;
            const obj = scn[prop];
            if (!obj) continue;

            for (const name of tryNames) {
              if (typeof obj[name] === 'function') {
                try {
                  // attempt with the string most plugins use
                  obj[name]('active');
                  console.log(`[QuestJournalForceActive] Switched via ${prop}.${name}('active')`);
                  switched = true;
                  break;
                } catch (err) {
                  // try peacefully with different friendly fallback name
                  try {
                    obj[name]('progress'); // some internals use 'progress' instead of 'active'
                    console.log(`[QuestJournalForceActive] Switched via ${prop}.${name}('progress')`);
                    switched = true;
                    break;
                  } catch (e2) {}
                }
              }
            }
            if (switched) break;
          }

          // Method C: fallback - find any window-like object and select first item
          if (!switched) {
            for (const k in scn) {
              const obj = scn[k];
              if (obj && typeof obj.refresh === 'function' && typeof obj.select === 'function') {
                try {
                  obj.refresh();
                  obj.select(0);
                  console.log(`[QuestJournalForceActive] Fallback selected index 0 on ${k}`);
                  switched = true;
                  break;
                } catch (e) {}
              }
            }
          }

          if (!switched) {
            // Friendly debugging log to help me help you if it fails
            console.warn('[QuestJournalForceActive] Could not auto-switch to Active tab. Scene keys:', Object.keys(scn));
            // If you paste that console message back to me I can craft a targeted patch.
          }
        }, 0);
      }
    } catch (ex) {
      console.error('[QuestJournalForceActive] Exception:', ex);
    }
  };
})();
